// Based on Derek Molloy's articles on Linux loadable kernel module (LKM) development. 
// http://www.derekmolloy.ie/ 

#include <linux/init.h>			  // Macros used to mark up functions e.g. __init __exit
#include <linux/module.h>			// Core header for loading LKMs into the kernel
#include <linux/device.h>			// Header types, macros, functions for the kernel
#include <linux/fs.h>				 // Header for the Linux file system support
#include <linux/uaccess.h>			 // Required for the copy to user function
#include <linux/slab.h>
#include <linux/mutex.h>	  // Required for the mutex functionality
#include <linux/gpio.h>

#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/sched.h>

#define THREAD_NAME "hdlc"
#define  DEVICE_NAME "bpqhdlc"	 ///< The device will appear at /dev/bpqchar using this value
#define  CLASS_NAME  "bpq"		  ///< The device class -- this is a character device driver

MODULE_LICENSE("GPL");				// The license type -- this affects available functionality
MODULE_AUTHOR("John Wiseman");	 // The author -- visible when you use modinfo
MODULE_DESCRIPTION("Interface for HDLC cards to LinBPQ");  // The description -- see modinfo
MODULE_VERSION("0.1");

// Everything should be static as this becomes part of kernel

static int majorNumber;				// Store the device number -- determined automatically
static short size_of_message;		// Used to remember the size of the string stored
static int numberOpens = 0;			// Counts the number of times the device is opened
	
static unsigned int gpioIRQ = 18;
static unsigned int irqNumber = -1;	// Allocated internally

static struct class*  bpqcharClass  = NULL; // The device-driver class struct pointer
static struct device* bpqcharDevice = NULL; // The device-driver device struct pointer

static DEFINE_MUTEX(bpqchar_mutex);	//	Prevent sharing device
static DEFINE_MUTEX(bpqchar_qlock);	//	Lock TX and RX Queues

static int dev_open(struct inode *, struct file *);
static int dev_release(struct inode *, struct file *);

static ssize_t dev_read(struct file *, char *, size_t, loff_t *);
static ssize_t dev_write(struct file *, const char *, size_t, loff_t *);

/**
 * Devices are represented as file structure in the kernel. The file_operations structure from
 * /linux/fs.h lists the callback functions that you wish to associated with your file operations
 * using a C99 syntax structure. char devices usually implement open, read, write and release calls
 */

static struct file_operations fops =
{
	.open = dev_open,
	.read = dev_read,
	.write = dev_write,
	.release = dev_release,
};

typedef struct MSG
{
	//	BASIC LINK LEVEL MESSAGE HEADER

	struct MSG * Next;
	size_t Len;
	unsigned char Data[330];

} *PMSG;

static PMSG TX_Q = NULL;		// From Appl to Device
static PMSG RX_Q = NULL;		// From Device to Appl

static void Q_ADD(PMSG * PQ, PMSG BUFF)
{
	PMSG P = *PQ;

	BUFF->Next = NULL;			// Clear chain in new buffer

	if (*PQ == NULL)			// Empty
	{
		*PQ = BUFF;
		return;
	}
	// Follow chain to end
	
	while (P->Next)
		P = P->Next;
	
	P->Next = BUFF;				// New one on end
	return;
}

static int Q_COUNT(PMSG Q)
{
	int count = 0;
	
	while (Q)
	{
		count++;
		Q = Q->Next;
	}
	return count;
}

static PMSG Q_REM(PMSG * PQ)
{
	PMSG Q = *PQ;

	// Get first in Q
	
	PMSG Msg = Q;
		
	if (Msg == NULL)
		return NULL;
		
	// At least one. Unchain it
	
	*PQ = Msg->Next;
	return Msg;
}

// Interrupt Handler

static irq_handler_t hdlc_irq_handler(unsigned int irq, void *dev_id, struct pt_regs *regs)
{ 
	// the actions that the interrupt should perform

	return (irq_handler_t) IRQ_HANDLED;
}

struct task_struct *task;

// TX Thread

int hdlc_thread(void *data)
{
//	struct task_struct *TSK;
//	struct sched_param PARAM = { .sched_priority = MAX_RT_PRIO };
//	TSK = current;

//	PARAM.sched_priority = THREAD_PRIORITY;
//	sched_setscheduler(TSK, SCHED_FIFO, &PARAM);

	while(1)
	{
//		printk(KERN_INFO "bpqhdlc: hdlc_thread running...");


	   PMSG Msg;

	   // Lock q access

	    mutex_lock(&bpqchar_qlock);
	    
	    Msg = Q_REM(&TX_Q);
	    if (Msg)
	    	Q_ADD(&RX_Q, Msg);
	    mutex_unlock(&bpqchar_qlock);

		schedule_timeout_interruptible(1 * HZ);
		if (kthread_should_stop())
			break;
	}
	return 0;
}

void hdlc_thread_init(void)
{
	int err;

	printk(KERN_INFO "bpqhdlc: starting thread...");
	
    task = kthread_create(hdlc_thread, NULL, "bpqhdlc_tx");
	
    if(IS_ERR(task)){ 
      printk("Unable to start kernel thread.\n");
      err = PTR_ERR(task);
      task = NULL;
      return;
    }
	
    wake_up_process(task); 

	printk(KERN_INFO "bpqhdlc: starting thread done.");
}

void hdlc_thread_exit(void)
{
	printk(KERN_INFO "bpqhdlc: stopping thread...");
	kthread_stop(task);
	printk(KERN_INFO "bpqhdlc: stopping thread done.");
}

/** @brief The LKM initialization function
 *  The static keyword restricts the visibility of the function to within this C file. The __init
 *  macro means that for a built-in driver (not a LKM) the function is only used at initialization
 *  time and that it can be discarded and its memory freed up after that point.
 *  @return returns 0 if successful
 */
static int __init bpqchar_init(void)
{
	int result;
	
	printk(KERN_INFO "bpqhdlc: Initializing\n");

	irqNumber = gpio_to_irq(gpioIRQ);

	if (irqNumber < 0)
	{
		printk(KERN_INFO "bpqhdlc: Failed to map GPIO to Interrupt: %d\n", irqNumber);
		return irqNumber;
	}
		
	printk(KERN_INFO "bpqhdlc: GPIO mapped to Interrupt: %d\n", irqNumber);

	// Try to dynamically allocate a major number for the device -- more difficult but worth it
	majorNumber = register_chrdev(0, DEVICE_NAME, &fops);
	if (majorNumber<0)
	{
		printk(KERN_ALERT "bpqhdlc failed to register a major number\n");
		return majorNumber;
	}
	
	printk(KERN_INFO "bpqhdlc: registered with major number %d\n", majorNumber);

	// Register the device class
	bpqcharClass = class_create(THIS_MODULE, CLASS_NAME);
	if (IS_ERR(bpqcharClass))
	{
		// Check for error and clean up if there is
	 
		unregister_chrdev(majorNumber, DEVICE_NAME);
		printk(KERN_ALERT "Failed to register device class\n");
		return PTR_ERR(bpqcharClass);	  // Correct way to return an error on a pointer
	}
	
	printk(KERN_INFO "bpqhdlc: device class registered\n");

	// Register the device driver
	
	bpqcharDevice = device_create(bpqcharClass, NULL, MKDEV(majorNumber, 0), NULL, DEVICE_NAME);
 
	if (IS_ERR(bpqcharDevice))
	{ 
		// Clean up if there is an error
	
		class_destroy(bpqcharClass);		// Repeated code but the alternative is goto statements
		unregister_chrdev(majorNumber, DEVICE_NAME);
		printk(KERN_ALERT "Failed to create the device\n");
		return PTR_ERR(bpqcharDevice);
	}
	
	printk(KERN_INFO "bpqhdlc: device class created\n"); // Made it! device was initialized
	

	result = request_irq(irqNumber,
		(irq_handler_t) hdlc_irq_handler,	// The pointer to the handler function below
		IRQF_TRIGGER_RISING,				// Interrupt on rising edge (button press, not release)
		"bpqhdlc_handler",					// Used in /proc/interrupts to identify the owner
		NULL);								// The *dev_id for shared interrupt lines, NULL is okay
 
	printk(KERN_INFO "bpqhdlc: The interrupt request result is: %d\n", result);

	mutex_init(&bpqchar_mutex);
	mutex_init(&bpqchar_qlock);
	
	hdlc_thread_init();

	return 0;
}

/** @brief The LKM cleanup function
 *  Similar to the initialization function, it is static. The __exit macro notifies that if this
 *  code is used for a built-in driver (not a LKM) that this function is not required.
 */
 
static void __exit bpqchar_exit(void)
{
	if (irqNumber >=0)				// Free IRQ if allocated
		free_irq(irqNumber, NULL);
	
	while (RX_Q)
		kfree(Q_REM(&RX_Q));

	while (TX_Q)
		kfree(Q_REM(&TX_Q));
		
	mutex_destroy(&bpqchar_mutex);							  // destroy the dynamically-allocated mutex		  
	device_destroy(bpqcharClass, MKDEV(majorNumber, 0)); // remove the device
	class_destroy(bpqcharClass);								 // remove the device clas
	unregister_chrdev(majorNumber, DEVICE_NAME);			// unregister the major number
	
	printk(KERN_INFO "bpqhdlc: Stopping tx thread\n");

	hdlc_thread_exit();

	printk(KERN_INFO "bpqhdlc: LKM Unloaded\n");
}

/** @brief The device open function that is called each time the device is opened
 *  This will only increment the numberOpens counter in this case.
 *  @param inodep A pointer to an inode object (defined in linux/fs.h)
 *  @param filep A pointer to a file object (defined in linux/fs.h)
 */
static int dev_open(struct inode *inodep, struct file *filep)
{
	// Try to acquire the mutex (returns 0 on fail)
	if(!mutex_trylock(&bpqchar_mutex))
	{						
		printk(KERN_ALERT "bpqhdlc: Device in use by another process");
		return -EBUSY;
	}
	
	numberOpens++;
	printk(KERN_INFO "bpqhdlc: Device has been opened %d time(s)\n", numberOpens);
	return 0;
}

/** @brief This function is called whenever device is being read from user space i.e. data is
 *  being sent from the device to the user. In this case is uses the copy_to_user() function to
 *  send the buffer string to the user and captures any errors.
 *  @param filep A pointer to a file object (defined in linux/fs.h)
 *  @param buffer The pointer to the buffer to which this function writes the data
 *  @param len The length of the b
 *  @param offset The offset if required
 */
static ssize_t dev_read(struct file *filep, char *buffer, size_t len, loff_t *offset)
{
	int error_count = 0;
	PMSG Msg;

	// Lock q access

	mutex_lock(&bpqchar_qlock);
	Msg = Q_REM(&RX_Q);
	mutex_unlock(&bpqchar_qlock);

	// copy_to_user has the format ( * to, *from, size) and returns 0 on success

	if (Msg)
	{
		size_of_message = Msg->Len;
		
		// silently discard excess data (caller should always supply big enough buffer
		
		if (size_of_message > len)
			size_of_message = len;
			
		error_count = copy_to_user(buffer, &Msg->Data, size_of_message);
		
		kfree(Msg);

		if (error_count == 0)
		{ 
			// success!
			printk(KERN_INFO "bpqhdlc: Sent %d characters to the user\n", size_of_message);
			return size_of_message; // clear the position to the start and return 0
		}
		else
		{
			printk(KERN_INFO "bpqhdlc: Failed to send %d characters to the user\n", error_count);
			return -EFAULT;		// Failed -- return a bad address message (i.e. -14)
		}
	}
	else
		return 0;		// Nothing available
}

/** @brief This function is called whenever the device is being written to from user space i.e.
 *  data is sent to the device from the user. The data is copied to the message[] array in this
 *  LKM using message[x] = buffer[x]
 *  @param filep A pointer to a file object
 *  @param buffer The buffer to that contains the string to write to the device
 *  @param len The length of the array of data that is being passed in the const char buffer
 *  @param offset The offset if required
 */
static ssize_t dev_write(struct file *filep, const char *buffer, size_t len, loff_t *offset)
{
	int ret;
 
	PMSG Msg;
	
 	Msg = (PMSG)kmalloc(sizeof(struct MSG), GFP_KERNEL);
 	
	if (len > 330)
		len = 330;			// silently discard excess data
		
	ret = copy_from_user(&Msg->Data, buffer, len);
	Msg->Len = len;

	// Lock queue access

	mutex_lock(&bpqchar_qlock);
	Q_ADD(&TX_Q, Msg);
	mutex_unlock(&bpqchar_qlock);
	
	printk(KERN_INFO "bpqhdlc: Received %zu bytes. Queue now %d\n", len, Q_COUNT(TX_Q));
	
	return len;
}

/** @brief The device release function that is called whenever the device is closed/released by
 *  the userspace program
 *  @param inodep A pointer to an inode object (defined in linux/fs.h)
 *  @param filep A pointer to a file object (defined in linux/fs.h)
 */
static int dev_release(struct inode *inodep, struct file *filep)
{
	mutex_unlock(&bpqchar_mutex);							 // release the mutex (i.e., lock goes up)
	printk(KERN_INFO "bpqhdlc: Device successfully closed\n");
	return 0;
}

/** @brief A module must use the module_init() module_exit() macros from linux/init.h, which
 *  identify the initialization function at insertion time and the cleanup function (as
 *  listed above)
 */
module_init(bpqchar_init);
module_exit(bpqchar_exit);

