
//	Boot loader for TNC-PI 9600 and maybe others
//	supports inmages up to 4K, but this could easily be extended if necessary

#define VERSION "0.0.0.2"

#define _CRT_SECURE_NO_DEPRECATE

#ifndef __GNUC__

#include "wingetopt.h"			// POSIX getopt for Windows

#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <stdio.h>
#include <windows.h>
#include <io.h>

#endif

#include <string.h>

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <time.h>
#include <sys/types.h>

#include <fcntl.h>
#include <signal.h>
#include <ctype.h>
#include <string.h>

#ifndef WIN32

#include <unistd.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <syslog.h>

#include <linux/i2c-dev.h>

#endif

#define G8BPQ_CRC	1

#define	SIZE		4096

#define FEND		0300	/* Frame End			(0xC0)	*/
#define FESC		0333	/* Frame Escape			(0xDB)	*/
#define TFEND		0334	/* Transposed Frame End		(0xDC)	*/
#define TFESC		0335	/* Transposed Frame Escape	(0xDD)	*/

#define TRUE 1
#define FALSE 0

/*
 * Keep these off the stack.
 */
 
static int address;
static char progname[80];

static unsigned char ibuf[SIZE];	/* buffer for input operations	*/


static char *usage_string	= "usage: pitnc_flash i2bus i2cdevice HexFileName\n"
								"Use i2cdevice = 0 for Serial Port\n";

#ifdef WIN32

#define ioctl ioctlsocket

int i2c_smbus_read_byte(HANDLE fd)
{
	return 0;
}

int i2c_smbus_write_byte(HANDLE fd, int chat)
{
	return 0;
}

VOID usleep(int Time)
{
	Sleep(Time / 1000);
}

#define open _open

#else
#define HANDLE int
#define BOOL int
#endif

HANDLE fd;
FILE * file;

int i2c = 0;

unsigned short NewCode[65536];
unsigned short OldCode[8192];
		
unsigned char EEPROM[64];

unsigned char Block[256];
unsigned char RXBlock[256];
unsigned short FlashAddr;


HANDLE OpenCOMPort(char * pPort, int speed);
int ReadCOMBlock(HANDLE fd, char * Block, int MaxLength);
BOOL WriteCOMBlock(HANDLE fd, char * Block, int BytesToWrite);

unsigned short reverse(unsigned short val)
{
	char x[2];
	char y[4];

	memcpy(x, &val, 2);
	y[0] = x[1];
	y[1] = x[0];

	memcpy(&val, y, 2);
	return val;
}

int WaitForResponse(unsigned char * RXBlock, int Len, int Timeout)
{
	// Return when we hav read Len chars, times out after Timeout mS

	int retval, n = 0;

	usleep(100000);			// Initial 100 mS

	Timeout -= 100;

	while (Timeout > 0)
	{
		if (i2c)
		{
			RXBlock[n] = i2c_smbus_read_byte(fd);		
			retval = RXBlock[0];
		
			if (retval == -1)	 		// Read failed		
  			{
				perror("poll failed");	 	
				exit(0);
			}
	
			if (retval == 0x0e)
				continue;

			n++;
		}
		else
			n += ReadCOMBlock(fd, &RXBlock[n], 256 - n);
	
		if (n >= Len)
			return n;

		usleep(10000);			// 10 mS
		Timeout -= 10;
	}

	return 0;
}

int ReadAndCheckEEPROM()
{
	//	Return TRUE if OK

	int n;
	
	Block[0] = 16;				// Read EEPROM
	WriteCOMBlock(fd, Block, 1);;

	n = WaitForResponse(EEPROM, 64, 10000);

	// EEPROM is at offest 0F000 in New Code

	if (n == 0)
	{
		printf("Timeout reading EEPROM\n");
		exit (1);
	}

	FlashAddr = 0xF000;

	for (n = 0; n < 64; n++)
	{
		if (EEPROM[n] != (NewCode[FlashAddr + n] & 0xff))
			return FALSE;
	}

	return TRUE;
}

								
int ReadAndCheck()
{
	//	Return TRUE if OK

	int n;
	
	FlashAddr = 0x200;			// Start of Progrma

ReadLoop:

	Block[0] = 2;				// Read
	Block[1] = FlashAddr >> 8;
	Block[2] = FlashAddr & 255;
	WriteCOMBlock(fd, Block, 3);;

	n = WaitForResponse((unsigned char *)&OldCode[FlashAddr], 64, 10000);

	if (n == 0)
	{
		printf("Timeout reading code\n");
		exit (1);
	}
							
	printf("%04x\r", FlashAddr);
	
	fflush(stdout);

	FlashAddr += 32;

	if (FlashAddr < 4096)
		goto ReadLoop;

	// Compare old code with new

	for (n = 0x200; n < 4096; n++)
	{
		if (reverse(OldCode[n]) != NewCode[n])
			return FALSE;
	}

	return TRUE;
}

int main(int argc, char *argv[])
{
	int retval, size;
	char i2cname [20];
	char serName[256] = "/dev/ttyAMA0";
	int fends = 0;
	int count = 0;    
	char * HexFile;
	int bus;
	int	ptr = 0;
	int sum = 0;
	int n;
	FILE *fp;
	char line[256];
	unsigned int Addr;
	unsigned int Segment;
	int RecLen;
	char work[16];
                        
	while ((size = getopt(argc, argv, ":v")) != -1) 
	{
		switch (size) 
		{
		case 'v':
			printf("%s\n", VERSION);
			return 1;
		case ':':
		case '?':
			fprintf(stderr, usage_string);
			return 1;
		}
	}

	// Need 3 params 
	
	if ((argc - optind) != 3 )
	{
		fprintf(stderr, usage_string);
		return 1;
	}
	
        bus = strtol(argv[optind], 0, 0);
        address = strtol(argv[optind + 1], 0, 0);
                
        if (address > 0)
           i2c = TRUE;
                                        
                                        	
	HexFile = argv[optind + 2];
 
	//	Open and read HexFile into NewCode

	for (n = 0; n < 65536; n ++)
		NewCode[n] = 0x3fff;			// Clear buffer

	if ((fp = fopen(HexFile, "r")) == NULL)
	{
		printf("Can't open Hex File %s\n", HexFile);
		exit (1);
	}

	while (fgets(line, 256, fp) != NULL)
	{
		if (line[0] != ':' || line[7] != '0')
		{
			printf("Invalid HEX record %s\n", line);
			exit (1);
		}

		if (line[8] == '1')
			break;					// End of file record

		if (line[8] == '4')
		{
			memcpy(work, line + 9, 4);
			work[4] = 0;
			Segment = strtol(work, 0, 16);
			continue;				// Segment Base
		}

		if (line[8] != '0')
			continue;
		
		// Extract Length and Addresss
		
		memcpy(work, line + 1, 2);
		work[2] = 0;
		RecLen = strtol(work, 0, 16)/2;

		memcpy(work, line + 3, 4);
		work[4] = 0;
		Addr = strtol(work, 0, 16);

		Addr = (Addr + (Segment << 16)) / 2;

		// Extract data. PIC uses 14 bit instructions, sent as two bytes

		for (n = 0; n < RecLen; n++)
		{
			memcpy(work, line + 9 + (n * 4), 4);
			work[4] = 0;
			NewCode[Addr + n] = reverse(strtol(work, 0, 16));
		}

	}

	if (i2c)
	{
		sprintf(i2cname, "/dev/i2c-%s", argv[optind]);
		fd = open(i2cname, O_RDWR);
                                              
		if (fd < 0)
		{
			fprintf(stderr, "%s: Cannot find i2c bus %s\n", progname, i2cname);
			exit(1);
		}
	
		address = strtol(argv[optind + 1], 0, 0);
 
#ifndef WIN32
		retval = ioctl(fd,  I2C_SLAVE, address);	
		if(retval == -1)
		{
			fprintf(stderr, "%s: Cannot open i2c device %x\n", progname, address);
			exit (1);
		}
#endif	
	}
	else
	{
		if (strlen(argv[optind]) > 3)
			strcpy(serName, argv[optind]);
		else
			if (bus > 0)
				sprintf(serName,"/dev/ttyO%d", bus);

		fd = OpenCOMPort(serName, 19200);                                                                                                       

	}

TryAgain:

	usleep(100000);
	
	// Send reset
	
	printf("Sending Reset Request\n");

	if (i2c)
	{
		i2c_smbus_write_byte(fd, FEND);
		i2c_smbus_write_byte(fd, 15);
		i2c_smbus_write_byte(fd, 2);
	}
	else
	{		
		Block[0] = FEND;
		Block[1] = FEND;
		Block[2] = 15;
		Block[3] = 2;
		
		WriteCOMBlock(fd, Block, 4);;
	}

	usleep(100000);			// 100 mS

	// Send Boot Request (FEND 55 AA) until program responds
			
	Block[0] = FEND;
	Block[1] = 0x55;
	Block[2] = 0xAA;
		
	WriteCOMBlock(fd, Block, 3);;

	n = WaitForResponse(RXBlock, 1, 5000);
		
	if (n == 0)
	{
		// Timeout

		goto TryAgain;
	}

	if (RXBlock[0] == 0xAA)
	{
		// TNC Wants to be reloaded
		
		int count = 0, ret;
		unsigned char FlashLine[68];
		int FlashTotal;
		unsigned char FlashChar, FlashSum;

		printf("Reading EEPROM from PIC\n");

		if (ReadAndCheckEEPROM())
		{
			printf("EEPROM is up to date\n");
		}
		else
		{
			printf("Updating EEPROM\n");

			FlashAddr = 0xF000;

			for (n = 0; n < 64; n++)
			{
				if (EEPROM[n] != (NewCode[FlashAddr + n] & 0xff))
				{
					Block[0] = 32;				// Write EEPROM
					Block[1] = NewCode[FlashAddr + n] & 0xff;
					Block[2] = n;
	
					WriteCOMBlock(fd, Block, 3);;

					ret = WaitForResponse(RXBlock, 1, 5000);

					if (ret == 0)
					{
						printf("Timeout updating EEPROM\n");
						exit (1);
					}
				}
			}
			
			printf("Verifying EEPROM\n");
			if (ReadAndCheckEEPROM())
			{
				printf("EEPROM ok\n");
			}
			else
			{
				printf("EEPROM update failed\n");
				exit(1);
			}	
		}
								
		fflush(stdout);

		printf("Reading Code from PIC\n");

		if (ReadAndCheck())
		{
			printf("\nCode is up to date\n");
			exit (0);
		}

		printf("\nCode is out of date, updating\nErasing...\n");

		Block[0] = 4;			// Erase		
		WriteCOMBlock(fd, Block, 1);;

		n = WaitForResponse(RXBlock, 1, 20000);

		if (n == 0)
		{
			printf("Timeout Erasing\n");
			exit (1);
		}

		printf("Erase Complete\nProgramming...\n", RXBlock[0]);

		// Write and non-blank blocks to the PIC

		// Start at 4K. Program Backwards, so it is easy to detect incomplete flash attempt

WriteLoop:

		FlashAddr -= 32;				
		
		FlashTotal = FlashSum = 0;

		FlashLine[0] = 1;		// Write
		FlashLine[1] = FlashAddr >> 8;
		FlashLine[2] = FlashAddr & 255;

		FlashSum ^= FlashLine[1];
		FlashSum ^= FlashLine[2];

		for (n = 0; n < 32; n++)
		{
			FlashChar = NewCode[FlashAddr + n] >> 8;
			FlashLine[2 * n + 3] = FlashChar;
			FlashSum ^= FlashChar;
			FlashTotal +=NewCode[FlashAddr + n];

			FlashChar = NewCode[FlashAddr + n] & 255;
			FlashLine[2 * n + 4] = FlashChar;
			FlashSum ^= FlashChar;
		}

		FlashLine[67] = FlashSum;

		if (FlashTotal != 32 * 0x3fff)			// Not Blank
		{
			printf("%04x\r", FlashAddr);
			fflush(stdout);

		Rewrite:
		
			WriteCOMBlock(fd, FlashLine, 68);;
			n = WaitForResponse(RXBlock, 1, 10000);

			if (n == 0)
			{
				printf("Timeout writing to Flash\n");
				exit (1);
			}
		
			if (RXBlock[0] == 0x55)
			{
				printf("Checksum Error transfering data to TNC\n");
				goto Rewrite;
			}
		}

		if (FlashAddr > 0x200)
			goto WriteLoop;

		//	Programming Complete. Verify it

		printf("\nProgramming Complete - Verifying Code\n");

		if (ReadAndCheck())
		{
			printf("Verified OK\n");
	
			Block[0] = 8;			// Load Complete		
			WriteCOMBlock(fd, Block, 1);;
			exit (0);
		}

	}
	exit (1);
}

#ifndef WIN32

static struct speed_struct
{
	int	user_speed;
	speed_t termios_speed;
} speed_table[] = {
	{300,         B300},
	{600,         B600},
	{1200,        B1200},
	{2400,        B2400},
	{4800,        B4800},
	{9600,        B9600},
	{19200,       B19200},
	{38400,       B38400},
	{57600,       B57600},
	{115200,      B115200},
	{-1,          B0}
};

#endif

#ifdef WIN32

HANDLE OpenCOMPort(char * pPort, int speed)
{
	char szPort[80];
	int fRetVal ;
	COMMTIMEOUTS  CommTimeOuts ;
	char buf[100];
	HANDLE fd;
	DCB dcb;

	// convert to \\.\COM or ports above 10 wont work

	sprintf(szPort, "\\\\.\\%s", pPort);

	printf("\nUsing Serial port %s\n", szPort);

	// open COMM device

	fd = CreateFile( szPort, GENERIC_READ | GENERIC_WRITE,
                  0,                    // exclusive access
                  NULL,                 // no security attrs
                  OPEN_EXISTING,
                  FILE_ATTRIBUTE_NORMAL,
                  NULL );

	if (fd == (HANDLE) -1)
	{
		int err = GetLastError();
		printf(" %s could not be opened Error %d\n ", pPort, err);
		exit (1);
	}


	// setup device buffers

	SetupComm(fd, 4096, 4096 ) ;

	// purge any information in the buffer

	PurgeComm(fd, PURGE_TXABORT | PURGE_RXABORT |
                                      PURGE_TXCLEAR | PURGE_RXCLEAR ) ;

	// set up for overlapped I/O

	CommTimeOuts.ReadIntervalTimeout = 0xFFFFFFFF ;
	CommTimeOuts.ReadTotalTimeoutMultiplier = 0 ;
	CommTimeOuts.ReadTotalTimeoutConstant = 0 ;
	CommTimeOuts.WriteTotalTimeoutMultiplier = 0 ;
//     CommTimeOuts.WriteTotalTimeoutConstant = 0 ;
	CommTimeOuts.WriteTotalTimeoutConstant = 500 ;
	SetCommTimeouts(fd, &CommTimeOuts ) ;

   dcb.DCBlength = sizeof( DCB ) ;

   GetCommState(fd, &dcb ) ;

   dcb.BaudRate = speed;
   dcb.ByteSize = 8;
   dcb.Parity = 0;
   dcb.StopBits = ONESTOPBIT;

	// setup hardware flow control

	dcb.fOutxDsrFlow = 0;
	dcb.fDtrControl = DTR_CONTROL_DISABLE ;

	dcb.fOutxCtsFlow = 0;
	dcb.fRtsControl = RTS_CONTROL_DISABLE ;

	// setup software flow control

   dcb.fInX = dcb.fOutX = 0;
   dcb.XonChar = 0;
   dcb.XoffChar = 0;
   dcb.XonLim = 100 ;
   dcb.XoffLim = 100 ;

   // other various settings

   dcb.fBinary = TRUE ;
   dcb.fParity = FALSE;

   fRetVal = SetCommState(fd, &dcb);

	if (fRetVal)
	{
		EscapeCommFunction(fd, SETDTR);
		EscapeCommFunction(fd, SETRTS);
	}
	else
	{
		printf(buf,"%s Setup Failed %d ", pPort, GetLastError());
		CloseHandle(fd);
		return 0;
	}

	return fd;
}

int ReadCOMBlock(HANDLE fd, char * Block, int MaxLength )
{
	BOOL       fReadStat ;
	COMSTAT    ComStat ;
	DWORD      dwErrorFlags;
	DWORD      dwLength;

	// only try to read number of bytes in queue

	ClearCommError(fd, &dwErrorFlags, &ComStat);

	dwLength = min((DWORD) MaxLength, ComStat.cbInQue);

	if (dwLength > 0)
	{
		fReadStat = ReadFile(fd, Block, dwLength, &dwLength, NULL) ;

		if (!fReadStat)
		{
		    dwLength = 0 ;
			ClearCommError(fd, &dwErrorFlags, &ComStat ) ;
		}
	}

   return dwLength;
}


BOOL WriteCOMBlock(HANDLE fd, char * Block, int BytesToWrite)
{
	BOOL        fWriteStat;
	DWORD       BytesWritten;
	DWORD       ErrorFlags;
	COMSTAT     ComStat;

	fWriteStat = WriteFile(fd, Block, BytesToWrite,
	                       &BytesWritten, NULL );

	if ((!fWriteStat) || (BytesToWrite != BytesWritten))
		ClearCommError(fd, &ErrorFlags, &ComStat);

	return TRUE;
}


#else
int OpenCOMPort(char * Port, int speed)
{
	char buf[100];

	//	Linux Version.

	int fd;
	int hwflag = 0;
	u_long param=1;
	struct termios term;
	struct speed_struct *s;

	printf("\nUsing Serial port %s\n", Port);

	if ((fd = open(Port, O_RDWR | O_NDELAY)) == -1)
	{
		printf("Device %s\n", Port);
		perror("Com Open Failed");
		exit (0);
	}

	// Validate Speed Param

	for (s = speed_table; s->user_speed != -1; s++)      
		if (s->user_speed == speed)
			break;

   if (s->user_speed == -1)
   {
	   fprintf(stderr, "tty_speed: invalid speed %d\n", speed);
	   exit (0);
   }
   
   if (tcgetattr(fd, &term) == -1)
   {
	   perror("tty_speed: tcgetattr");
	   exit (0);
   }

   	cfmakeraw(&term);
	cfsetispeed(&term, s->termios_speed);
	cfsetospeed(&term, s->termios_speed);

	if (tcsetattr(fd, TCSANOW, &term) == -1)
	{
		perror("tty_speed: tcsetattr");
		return FALSE;
	}

	ioctl(fd, FIONBIO, &param);

	return fd;
}

int ReadCOMBlock(int fd, char * Block, int MaxLength)
{
	int Length;
	
	Length = read(fd, Block, MaxLength);

	if (Length < 0)
	{
		if	(errno != 11)					// Would Block
		{
			perror("read");
			printf("%d\n", errno);
		}
		return 0;
	}

	return Length;
}
int WriteCOMBlock(HANDLE fd, char * Block, int BytesToWrite)
{
	return write(fd, Block, BytesToWrite);
	return TRUE;
}

#endif
